## App Basic Architecture

### Layout Function

DevFlow App basically contains two parts in code: layout function and callbacks.

In Basic App, we can see the layout function is defined as a method of a class, and the class is decorated with ```@mark_create_layout```. This is a special decorator that will tell DevFlow to create a layout function from the method. The layout function will be called when the app is loaded, and the returned layout will be rendered in the app.

**WARNING**: layout function must return a ```mui.FlexBox``` instance. ```mui.HBox```/```mui.VBox``` are functions that return ```mui.FlexBox``` instance.

```Python 
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Typography("Hello World"),
        ])
```

### Events

Event handling in devflow is callback-based. You can register a callback function to an event, and the callback function will be called when the event is triggered.

Many components usually have events. For example, a button has a click event. we can firstly define a callback method for this button, then provide the callback method to the button. When the button is clicked, the callback method will be called.

Most of components with events contains a ```callback``` in their constructor. Their arguments are simplified, e.g. ```on_click``` takes no arguments, ```on_change``` takes one argument ```value```, etc.


```Python 
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import asyncio

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Button("A Button", self._on_button_clicked),
            mui.Button("A Async Button", self._on_button_clicked_async),
        ])

    def _on_button_clicked(self):
        print("Button Clicked")

    async def _on_button_clicked_async(self):
        await asyncio.sleep(0.5)
        print("Button Clicked After 0.5 Second")
```


In example above, the callback function can be both async and sync.

### Send events to Frontend

Most of components use a "reactive way" to update props, i.e. when a prop is changed, the component will be re-rendered. For example, we can change the text of a button by changing the ```text``` prop of the button.

Unlike callbacks, if we want to send events to frontend, we must use async code, because ```tensorpc.flow``` is a async UI framework.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.button = mui.Button("A Button", self._on_button_clicked)
        self.flag = True
        return mui.VBox([
            self.button,
        ])

    async def _on_button_clicked(self):
        print("Button Clicked")
        self.flag = not self.flag
        update_event = self.button.update_event(name="A Button" if self.flag else "A Outlined Button", 
                                                variant="text" if self.flag else "outlined")
        await self.button.send_and_wait(update_event)

```

In example above, the ```update_event``` means update some props of ```Button```. when your cursor hover on method ```update_event```, you can see all available props of the button. After create update event, we call ```send_and_wait``` to send message to relay (master machine), wait for send finish, then relay will send message to frontend via websocket.

If you don't want to wait for send finish (usually performance reason), you can use ```put_app_event``` instead of ```send_and_wait```.

### Props 
A component can have lots of props. For example, a button has ```text```, ```variant```, ```color```, ```disabled```, ```size``` and so on. You can change these props to change the appearance of the button.

**WRANING**: devflow props use camel case, not snake case. e.g. ```x.prop(flexShrink=...)```. other python code use standard snake case.

In devflow, each component contains two functions: ```prop``` and ```update_event```.

* ```prop```: set component prop in backend, won't affact frontend prop. usually used in layout function or init function. this function returns the component itself, so you can chain multiple ```prop``` calls.

* ```update_event```: create update event that can be sent to frontend to update frontend prop. usually used in callback function. the event can be sent via ```send_and_wait``` or ```put_app_event```. this function will also perform prop set like ```prop```.

Props passed to ```prop``` or ```update_event``` will be validated by ```pydantic``` package. if the prop is invalid, an exception will be raised. keep in mind that if you use ```setattr``` to set prop, the prop won't be validated. e.g. ```x.props.flexShrink = None``` won't be validated, but ```x.prop(flexShrink=None)``` will be validated. **ALWAYS** use ```prop``` to set props.

**Note** use vscode (pyright) instead of pycharm. pyright can recognize the type of ```prop``` and ```update_event```, so you can get auto-completion and type check.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.button = mui.Button("A Text Button", self._on_button_clicked)
        self.button.prop(variant="text")
        self.flag = True
        return mui.VBox([
            self.button,
            mui.Button("A Outlined Button").prop(variant="outlined"),
            mui.Button("A Contained Button").prop(variant="contained"),
        ])

    async def _on_button_clicked(self):
        print("Button Clicked")
        self.flag = not self.flag
        update_event = self.button.update_event(name="A Text Button" if self.flag else "A Outlined Button", 
                                                variant="text" if self.flag else "outlined")
        await self.button.send_and_wait(update_event)

```

